'''
Created on 7 oct. 2013

@author: Frederic Ravetier

Modified by Thierry Bousquet - 16 nov 2013 : add support for 3 thermometers
'''

import requests
import json
import hashlib
import logging
import ConfigParser
import sys

class Zipabox:
    """Give methods to access to the data of a Zipabox thru the cloud of Zipato.

    Provide init, login and logout methods.
    The configuration file contains the personal informations in order to separate the code from the personnal informations.
    The password is stored encrypted, so someone can re-use it but can not know the password (that may be used is some others applications...
    
    It provides a method to get values from meters depending on feed_id define in the configuration file

    Basic Usage::

      >>> zipabox = Zipabox()
      >>> zipabox.processMetersToSense()
    """
    
    def __init__(self):
        #### INIT CONFIG ####
        self.configFile = open(r'config.ini')
        self.config = ConfigParser.ConfigParser()
        self.config.readfp(self.configFile)
        #config.read(".\config.ini")
        try:
            loglevel=self.config.get("Development",'LogLevel')
        except (KeyError, ConfigParser.NoOptionError, ConfigParser.NoSectionError):
            loglevel="INFO"        
        self.logger = logging.getLogger("Zipabox")
        if loglevel == "DEBUG":
            logging.basicConfig(level=logging.DEBUG)
        else:
            logging.basicConfig(level=logging.INFO)
        self.logger.info('Starting Processus...')
           
    def initLoginZipato(self):
        """Initialize the connexion to Zipabox. It initializes the session.
        :return the data in JSON that contains the nonce. Exists if an error occured.
        """
        #### READ CONFIG ABOUT Authentication ####
        try:
            self.username = self.config.get("Authentication",'Login')
            if self.username == '': 
                self.logger.error('Login is missing in the config file')
                sys.exit(1)
            try:
                self.passwordSha1 =  self.config.get("Authentication",'PasswordSHA1')
            except (KeyError, ConfigParser.NoOptionError) as e:
                self.passwordSha1 = ''
            #Check if this encrypted password exists. If yes, we use it, otherwise we generate it from the normal password.
            if  self.passwordSha1 == '':
                password =  self.config.get("Authentication",'Password')
                if password == '':
                    self.logger.error('Password is missing in the config file')
                    sys.exit(1)
                self.passwordSha1 = hashlib.sha1(password.encode('utf-8')).hexdigest()
                self.logger.info('You should set the SHA1 password in the config file : '+self.passwordSha1)
                self.config.set("Authentication",'PasswordSHA1',self.passwordSha1)
                self.config.set("Authentication",'Password','')
                self.config.write(open(r'config.ini', 'w'))
        except (KeyError, ConfigParser.NoOptionError) as e:
            self.logger.error('An error occured during the initialization of the config File : '+e)                                 
        #### INIT LOGIN ####
        try :
            self.uriZipato = self.config.get("NETWORK",'protocol')+'://'+self.config.get("NETWORK",'ip')+':'+self.config.get("NETWORK",'port')
        except (KeyError, ConfigParser.NoOptionError) as e:            
            self.uriZipato = 'https://my.zipato.com:443'
            self.logger.info('Missing NETWORK configuration, cloud configuration by default : '+str(e))            
        path = '/zipato-web/rest/user/init'
        self.session = requests.Session()
        self.logger.debug('Init login process to ' + self.uriZipato+path)
        response = self.session.get(self.uriZipato+path)
        if (response.status_code != 200) :
            self.logger.error('Error during init on Zipato : ', response)
            sys.exit(1)
        data = response.json()
        self.logger.debug("response: "+str(response) + ' - ' + str(data))
        return data
        
    def loginZipato(self,data):
        """Does the login on the Zipabox. The initLogin should be done before
        :param data : the data in JSON, it should contains the nonce
        :return 0 if OK; exists otherwise.
        """
        #### BUILD TOKEN ####
        #token = SHA1(nonce + SHA1(password))
        #sidKey = 'jsessionid'
        nonceKey  = 'nonce'
        nonceValue = data[nonceKey]
        #self.logger.debug(sidKey+": "+data[sidKey])
        self.logger.debug(nonceKey +": "+nonceValue)
        token = hashlib.sha1((nonceValue + self.passwordSha1).encode('utf-8')).hexdigest()
        
        #### LOGIN ####
        path = '/zipato-web/json/Login'
        parameters = {'username': self.username, 'password': token, 'method': 'SHA1'}
        self.logger.debug('Login to ' + self.uriZipato+path + ', params : '+str(parameters))
        response = self.session.post(self.uriZipato+path,params=parameters)
        if (response.status_code != 200) :
            self.logger.error('Error during authentication on Zipato : ', response)
            sys.exit(1)
        data = response.json()
        self.logger.debug("response: "+str(response) + ' - ' + str(data))
        success = data['success']
        self.logger.debug('Login Success: '+str(success))
        if (str(success) != 'True') :
            self.logger.error('Unable to login : ', data)
            sys.exit(1)
        return 0
    
    def logoutZipato(self):
        """Does the logout on the Zipabox.
        :return 0 if OK; exists otherwise.
        """
        path = '/zipato-web/rest/user/logout'
        response = self.session.get(self.uriZipato+path)
        if (response.status_code != 200) :
            self.logger.error('Error during logout on Zipato : ', response)
            sys.exit(1)
        data = response.json()
        self.logger.debug("response: "+str(response) + ' - ' + str(data))
        success = data['success']
        self.logger.debug('Logout Success: '+str(success))
        if (str(success) != 'True') :
            self.logger.error('Unable to logout : ', data)
            sys.exit(1)
        return 0

    def getFeedId(self,typeDevice,deviceName,deviceId):
        """Get the feedId from the config file
        :param deviceName : the device name, not used at this time
        :param deviceId : the deviceId from the zipabox
        :return the feed_id matching the device, return an empty string '' if not found
        """
        feed_id = ''
        try :
            feed_id = self.config.get(typeDevice, deviceName)
        except (KeyError, ConfigParser.NoOptionError) :
            try:
                feed_id = self.config.get(typeDevice, deviceId)
            except (KeyError, ConfigParser.NoOptionError) :
                feed_id = ''
        return feed_id
    
    def getTemperature1(self,attributeValue,name, deviceId):
        #Apply a filter on TEMPERATURE                
        if attributeValue ['definition']['name'] == "TEMPERATURE":
            value = float(attributeValue['value']) 
            value = "{0:.2f}".format(value)           
            feed_id = self.getFeedId("TEMPERATURE",name,deviceId)            
            if feed_id != '':
                self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : '+str(value))
                parametersTmp = {'feed_id': str(feed_id), 'value': str(value)}
                return parametersTmp
            else:
                self.logger.info('NO FEED for Device : ' + deviceId + '(' + name + ') : '+str(value))

    def getTemperature2(self,attributeValue,name, deviceId):
        #Apply a filter on TEMPERATURE2                
        if attributeValue ['definition']['name'] == "TEMPERATURE":
            value = float(attributeValue['value']) 
            value = "{0:.2f}".format(value)           
            feed_id = self.getFeedId("TEMPERATURE",name,deviceId)            
            if feed_id != '':
                self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : '+str(value))
                parametersTmp = {'feed_id': str(feed_id), 'value': str(value)}
                return parametersTmp
            else:
                self.logger.info('NO FEED for Device : ' + deviceId + '(' + name + ') : '+str(value))                
    def getTemperature3(self,attributeValue,name, deviceId):
        #Apply a filter on TEMPERATURE3                
        if attributeValue ['definition']['name'] == "TEMPERATURE":
            value = float(attributeValue['value']) 
            value = "{0:.2f}".format(value)           
            feed_id = self.getFeedId("TEMPERATURE",name,deviceId)            
            if feed_id != '':
                self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : '+str(value))
                parametersTmp = {'feed_id': str(feed_id), 'value': str(value)}
                return parametersTmp
            else:
                self.logger.info('NO FEED for Device : ' + deviceId + '(' + name + ') : '+str(value))

    def getCumulativeConsumption(self,attributeValue,name, deviceId):
        #Apply a filter on CUMULATIVE_CONSUMPTION                
        if attributeValue ['definition']['name'] == "CUMULATIVE_CONSUMPTION":
            value = float(attributeValue['value']) 
            value = "{0:.2f}".format(value)           
            feed_id = self.getFeedId("CUMULATIVE_CONSUMPTION",name,deviceId)            
            if feed_id != '':
                self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : '+str(value))
                parametersTmp = {'feed_id': str(feed_id), 'value': str(value)}
                return parametersTmp
            else:
                self.logger.info('NO FEED for Device : ' + deviceId + '(' + name + ') : '+str(value))      
                
    def getCurrentConsumption(self,attributeValue,name, deviceId):
        #Apply a filter on CUMULATIVE_CONSUMPTION                
        if attributeValue ['definition']['name'] == "CURRENT_CONSUMPTION":
            value = float(attributeValue['value']) 
            value = "{0:.2f}".format(value)           
            feed_id = self.getFeedId("CURRENT_CONSUMPTION",name,deviceId)            
            if feed_id != '':
                self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : '+str(value))
                parametersTmp = {'feed_id': str(feed_id), 'value': str(value)}
                return parametersTmp
            else:
                self.logger.info('NO FEED for Device : ' + deviceId + '(' + name + ') : '+str(value))              

    def getMeters(self):   
        """Get the Meters from the Zipabox. At this time it gets only the temperature of the devices in the config file.
        :return the data in JSON with feed_id and values, return [] if nothing
        """
        #Browse the meters
        #/zipato-web/rest/meters/
        path = '/zipato-web/rest/meters/'
        self.logger.debug('Get meters: ' + self.uriZipato+path)
        response = self.session.get(self.uriZipato+path)
        if (response.status_code != 200) :
            self.logger.error('Unable to get meters from zipato : ', path, response)
            sys.exit(1)
        data = response.json()
        self.logger.debug("response: "+str(response) + ' - ' + str(data))
        parameters = []
        for deviceId, deviceValue in data.items():
            name=deviceValue['name']   
            try :
                offline=deviceValue['offline']
            except (KeyError) :
                offline = False
            attributes=deviceValue['attributes']
            for attributeKey, attributeValue in attributes.items():
                self.logger.debug('name = ' +attributeKey + ':' + str(attributeValue))
                parametersTmp = self.getTemperature1(attributeValue,name,deviceId)
                if parametersTmp:
                    parameters.append(parametersTmp)
                parametersTmp = self.getTemperature2(attributeValue,name,deviceId)
                if parametersTmp:
                    parameters.append(parametersTmp)
                parametersTmp = self.getTemperature3(attributeValue,name,deviceId)
                if parametersTmp:
                    parameters.append(parametersTmp)
                parametersTmp = self.getCumulativeConsumption(attributeValue,name,deviceId)
                if parametersTmp:
                    parameters.append(parametersTmp)   
                parametersTmp = self.getCurrentConsumption(attributeValue,name,deviceId)
                if parametersTmp:
                    parameters.append(parametersTmp)                        
                feed_id = self.getFeedId("Sen.se",'offline_feed',deviceId)            
                if offline and feed_id != '':
                    self.logger.info('feed_id = '+feed_id + ' - Device : ' + deviceId + '(' + name + ') : Offline : '+str(offline))
                    try:                    
                        parametersTmp = {'feed_id': str(feed_id), 'value': name.encode('utf-8')}
                        #Do not insert doubles, some devices are able to manage more than one 'device'
                        if parameters.count(parametersTmp) == 0 :
                            parameters.append(parametersTmp)
                    except (UnicodeEncodeError) as e:
                        self.logger.error(e)
                                        
        #parameters = {'feed_id': str(feed_id), 'value': str(value)}
        return parameters
    
    def sendToSense(self,data):
        """Send the data to Sen.se. The sens.se key is get from the config file
        :param data : the data to send. It should be a list of dict
        :return 0 if success, 1 otherwise. return 0 if there is no data to send
        """
        self.logger.debug("data : "+str(data))
        if data:
            senseKey = ''
            uriSense = "http://api.sen.se";
            if senseKey == '':   
                senseKey = self.config.get("Sen.se",'sense_key')             
            path = '/events/?sense_key='+senseKey
            headersSense = {'Content-type': 'application/json'}
            self.logger.debug('Send data ' + uriSense+path + ', params : '+json.dumps(data))
            response = requests.post(uriSense+path,json.dumps(data),headers=headersSense)
            if (response.status_code != 200) :
                self.logger.error('Error sending data to sen.se : '+ str(response))            
                self.logger.debug("response: "+str(response))
                return 1
            else:
                self.logger.info('Data sent to Sen.se')
            return 0
        return 0
    
    def processMetersToSense(self):
        """It does the whole process: init, login, getTemperatures, sendToSense.
        """
        data = self.initLoginZipato()
        result = self.loginZipato(data)
        if result == 0:
            try:
                parameters = self.getMeters()
                #self.logger.info(str(parameters))
                self.sendToSense(parameters)            
            finally:
                self.logoutZipato()        
        
             

if __name__ == '__main__':
    zipabox = Zipabox()
    zipabox.processMetersToSense()
    
